'use strict';

var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function (obj) { return typeof obj; } : function (obj) { return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj; };

/* --------------------------------------------------------------
 cart_handler.js 2025-05-13
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2025 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Component for handling the add to cart and wishlist features
 * at the product details and the category listing pages. It cares
 * for attributes, properties, quantity and all other
 * relevant data for adding an item to the basket or wishlist
 */
gambio.widgets.module('cart_handler', ['hooks', 'form', 'xhr', 'loading_spinner', gambio.source + '/libs/events', gambio.source + '/libs/modal.ext-magnific', gambio.source + '/libs/modal'], function (data) {

	'use strict';

	// ########## VARIABLE INITIALIZATION ##########

	var $this = $(this),
	    $body = $('body'),
	    $window = $(window),
	    busy = false,
	    ajax = null,
	    timeout = 0,
	    previousModifiers = {},
	    defaults = {
		// AJAX "add to cart" URL
		addCartUrl: 'shop.php?do=Cart/BuyProduct',
		// AJAX "add to cart" URL for customizer products
		addCartCustomizerUrl: 'shop.php?do=Cart/Add',
		// AJAX URL to perform a value check
		checkUrl: 'shop.php?do=CheckStatus',
		// AJAX URL to perform the add to wishlist
		wishlistUrl: 'shop.php?do=WishList/Add',
		// Submit URL for price offer button
		priceOfferUrl: 'gm_price_offer.php',
		// Submit method for price offer
		priceOfferMethod: 'get',
		// Selector for the cart dropdown
		dropdown: '#head_shopping_cart',
		// "Add to cart" buttons selectors
		cartButtons: '.js-btn-add-to-cart',
		// "Wishlist" buttons selectors
		wishlistButtons: '.btn-wishlist',
		// "Price offer" buttons selectors
		priceOfferButtons: '.btn-price-offer',
		// Selector for the attribute fields
		attributes: '.js-calculate',
		// Selector for product property
		productOptions: '.modifier-group .modifier-content .modifier-item',
		productOptionField: '.hidden-input',
		// Selector for the quantity
		quantity: '.js-calculate-qty',
		// URL where to get the theme for the dropdown
		tpl: null,
		// Show attribute images in product images swiper (if possible)
		// -- this feature is not supported yet --
		attributImagesSwiper: false,
		// Trigger the attribute images to this selectors
		triggerAttrImagesTo: '#product_image_swiper, #product_thumbnail_swiper, ' + '#product_thumbnail_swiper_mobile',
		// Class that gets added to the button on processing
		processingClass: 'loading',
		// Duration for that the success or fail class gets added to the button
		processingDuration: 2000,
		// AJAX response content selectors
		selectorMapping: {
			buttons: '.shopping-cart-button',
			giftContent: '.gift-cart-content-wrapper',
			giftLayer: '.gift-cart-layer',
			shareContent: '.share-cart-content-wrapper',
			shareLayer: '.share-cart-layer',
			hiddenOptions: '#cart_quantity .hidden-options',
			message: '.global-error-messages',
			messageCart: '.cart-error-msg',
			messageHelp: '.help-block',
			modelNumber: '.model-number',
			modelNumberText: '.model-number-text',
			price: '.current-price-container',
			modifiersForm: '.modifiers-selection',
			quantity: '.products-quantity-value',
			quantityInfo: '.products-quantity',
			ribbonSpecial: '.ribbon-special',
			shippingInformation: '#shipping-information-layer',
			shippingTime: '.products-shipping-time-value',
			shippingTimeImage: '.img-shipping-time img',
			totals: '#cart_quantity .total-box',
			weight: '.products-details-weight-container span',
			abroadShippingInfo: '.abroad-shipping-info'
		},
		page: 'product-listing'
	},
	    options = $.extend(true, {}, defaults, data),
	    module = {},
	    mobile = $(window).width() <= 767;

	// ########## HELPER FUNCTIONS ##########

	/**
  * Helper function that updates the button
  * state with an error or success class for
  * a specified duration
  * @param   {object}        $target         jQuery selection of the target button
  * @param   {string}        state           The state string that gets added to the loading class
  * @private
  */
	var _addButtonState = function _addButtonState($target, state) {
		var timer = setTimeout(function () {
			$target.removeClass(options.processingClass + ' ' + options.processingClass + state);
		}, options.processingDuration);

		$target.data('timer', timer).addClass(options.processingClass + state);
	};

	/**
  * Helper function to set the messages and the
  * button state.
  * @param       {object}    data                Result form the ajax request
  * @param       {object}    $form               jQuery selecion of the form
  * @param       {boolean}   disableButtons      If true, the button state gets set to (in)active
  * @param       {boolean}   showNoCombiMesssage If true, the error message for missing property combination selection will be displayed
  * @private
  */
	var _stateManager = function _stateManager(data, $form, disableButtons, showNoCombiSelectedMesssage) {

		// Save the current changed modifier
		var lastFocusedElement = $(':focus');
		var lastFocusedElementID = void 0;
		if (lastFocusedElement.hasClass('modifier-item')) {
			lastFocusedElementID = '[data-value="' + lastFocusedElement.data('value') + '"]';
		} else {
			var lastFocusedElementName = lastFocusedElement.attr('name');
			var isLastFocusedElementRadio = lastFocusedElement.attr('type') === 'radio';
			lastFocusedElementID = isLastFocusedElementRadio ? '[name="' + lastFocusedElementName + '"]:checked' : '[name="' + lastFocusedElementName + '"]';
		}

		// Remove the attribute images from the common content
		// so that it doesn't get rendered anymore. Then trigger
		// an event to the given selectors and deliver the
		// attrImages object
		if (options.attributImagesSwiper && data.attrImages && data.attrImages.length) {
			delete data.content.images;
			$(options.triggerAttrImagesTo).trigger(jse.libs.theme.events.SLIDES_UPDATE(), { attributes: data.attrImages });
		}

		// Set the messages given inside the data.content object
		$.each(data.content, function (i, v) {
			var $element = $body.hasClass('page-product-info') ? $this.find(options.selectorMapping[v.selector]) : $form.parent().find(options.selectorMapping[v.selector]);

			if ((!showNoCombiSelectedMesssage || v.value === '') && i === 'messageNoCombiSelected') {
				return true;
			}

			switch (v.type) {
				case 'hide':
					if (v.value === 'true') {
						$element.hide();
					} else {
						$element.show();
					}
					break;
				case 'html':
					$element.html(v.value);
					break;
				case 'attribute':
					$element.attr(v.key, v.value);
					break;
				case 'replace':
					if (v.value) {
						$element.replaceWith(v.value);
					} else {
						$element.addClass('hidden').empty();
					}
					break;
				default:
					$element.text(v.value);
					break;
			}
		});

		// Dis- / Enable the buttons
		if (disableButtons) {
			var $buttons = $form.find(options.cartButtons);
			if (data.success) {
				$buttons.removeClass('inactive');
				$buttons.removeClass('btn-inactive');
				$buttons.prop("disabled", false);
			} else {
				$buttons.addClass('inactive');
				$buttons.addClass('btn-inactive');
				$buttons.prop("disabled", true);
			}
		}

		if (data.content.message) {
			var $errorField = $form.find(options.selectorMapping[data.content.message.selector]);
			if (data.content.message.value) {
				$errorField.removeClass('hidden').show();
			} else {
				$errorField.addClass('hidden').hide();

				if (showNoCombiSelectedMesssage && data.content.messageNoCombiSelected !== undefined && data.content.messageNoCombiSelected) {
					if (data.content.messageNoCombiSelected.value) {
						$errorField.removeClass('hidden').show();
					} else {
						$errorField.addClass('hidden').hide();
					}
				}
			}
		}

		$window.trigger(jse.libs.theme.events.STICKYBOX_CONTENT_CHANGE());
	};

	/**
  * Helper function to send the ajax
  * On success redirect to a given url, open a layer with
  * a message or add the item to the cart-dropdown directly
  * (by triggering an event to the body)
  * @param       {object}      data      Form data
  * @param       {object}      $form     The form to fill
  * @param       {string}      url       The URL for the AJAX request
  * @private
  */
	var _addToSomewhere = function _addToSomewhere(data, $form, url, $button) {
		function callback() {
			jse.libs.xhr.post({ url: url, data: data }, true).done(function (result) {
				try {
					// Fill the page with the result from the ajax
					_stateManager(result, $form, false);

					// If the AJAX was successful execute
					// a custom functionality
					if (result.success) {
						switch (result.type) {
							case 'url':
								if (result.url.substr(0, 4) !== 'http') {
									location.href = jse.core.config.get('appUrl') + '/' + result.url;
								} else {
									location.href = result.url;
								}

								break;
							case 'dropdown':
								$body.trigger(jse.libs.theme.events.CART_UPDATE(), [true]);
								break;
							case 'layer':
								jse.libs.theme.modal.info({ title: result.title, content: result.msg });
								break;
							default:
								break;
						}
					}
				} catch (ignore) {}
				_addButtonState($button, '-success');
			}).fail(function () {
				_addButtonState($button, '-fail');
			}).always(function () {
				// Reset the busy flag to be able to perform
				// further AJAX requests
				busy = false;
			});
		}

		if (!busy) {
			// only execute the ajax
			// if there is no pending ajax call
			busy = true;

			jse.libs.hooks.execute(jse.libs.hooks.keys.shop.cart.add, data, 500).then(callback).catch(callback);
		}
	};

	// ########## EVENT HANDLER ##########

	/**
  * Hide the busy/loading-animation on the "add to cart" button.
  *
  * @param $form
  * @private
  */
	var _removeAddToCartThrobbler = function _removeAddToCartThrobbler($form) {
		var $btn = $this.find("[name=btn-add-to-cart]");
		var $btnFake = $this.find(".btn-add-to-cart-fake");
		if ($btnFake) {
			$btnFake.hide().prop("disabled", false).find('.throbbler').remove();
		}
		if ($btn) {
			$btn.show();
		}
	};

	/**
 * Handler for the submit form / click
 * on "add to cart" & "wishlist" button.
 * It performs a check on the availability
 * of the combination and quantity. If
 * successful it performs the add to cart
 * or wishlist action, if it's not a
 * "check" call
 * @param       {object}    e      jQuery event object
 * @private
 */
	var _submitHandler = function _submitHandler(e) {
		if (e) {
			e.preventDefault();
		}

		var $self = $(this),
		    $form = $self.is('form') ? $self : $self.closest('form'),
		    customizer = $form.hasClass('customizer'),
		    properties = !!$form.find('.properties-selection-form').length,
		    module = properties ? '' : '/Attributes',
		    showNoCombiSelectedMesssage = e && e.data && e.data.target && e.data.target !== 'check';

		if ($form.length) {

			// Show properties overlay
			// to disable user interaction
			// before markup replace
			if (properties) {
				$this.addClass('loading');
			}

			if ($self.is('select')) {
				var price = $self.find(":selected").attr('data-price');
				$self.parents('.modifier-group').find('.selected-value-price').text(price);
			}

			var getGalleryHash = $('#current-gallery-hash').val();
			$form.find('#update-gallery-hash').val(getGalleryHash);

			var formdata = jse.libs.form.getData($form, null, true);
			formdata.target = e && e.data && e.data.target ? e.data.target : 'check';
			formdata.isProductInfo = $form.hasClass('product-info') ? 1 : 0;

			// Abort previous check ajax if
			// there is one in progress
			if (ajax && e) {
				ajax.abort();
			}

			// Add processing-class to the button
			// and remove old timed events
			if (formdata.target !== 'check') {
				var timer = $self.data('timer');
				if (timer) {
					clearTimeout(timer);
				}

				$self.removeClass(options.processingClass + '-success ' + options.processingClass + '-fail').addClass(options.processingClass);
			}

			formdata.previousModifiers = previousModifiers;

			ajax = jse.libs.xhr.get({
				url: options.checkUrl + module,
				data: formdata
			}, true).done(function (result) {
				_stateManager(result, $form, true, showNoCombiSelectedMesssage);
				$this.removeClass('loading');

				// Check if the gallery images changed
				if (formdata.target === 'check' && result.content.imageGallery.trim() !== '' && result.content.replaceGallery === true && formdata.isProductInfo === 1) {
					var loadingSpinner = jse.libs.loading_spinner.show($('.product-info-stage'), 9999);

					var swipers = [$('#product_image_swiper'), $('#product_thumbnail_swiper'), $('#product_thumbnail_swiper_mobile')];

					var _iteratorNormalCompletion = true;
					var _didIteratorError = false;
					var _iteratorError = undefined;

					try {
						for (var _iterator = swipers[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
							var element = _step.value;

							var instance = element.swiper();

							if ((typeof instance === 'undefined' ? 'undefined' : _typeof(instance)) !== 'object') {
								continue;
							}

							instance.destroy(true, true);
							element.off().remove();
						}
					} catch (err) {
						_didIteratorError = true;
						_iteratorError = err;
					} finally {
						try {
							if (!_iteratorNormalCompletion && _iterator.return) {
								_iterator.return();
							}
						} finally {
							if (_didIteratorError) {
								throw _iteratorError;
							}
						}
					}

					$('#image-collection-container').html(result.content.imageGallery);
					$('#product_image_layer').html(result.content.imageModal);

					gambio.widgets.init($('.product-info-content')).done(function () {
						jse.libs.loading_spinner.hide(loadingSpinner);
					});
				} else if (formdata.target === 'check' && result.content.imageGallery.trim() === '' && result.content.replaceGallery === true) {
					$('#image-collection-container').html(result.content.imageGallery);
					$('#product_image_layer').html(result.content.imageModal);
				}

				if (result.success) {
					var event = null,
					    url = null;

					switch (formdata.target) {
						case 'wishlist':
							if (customizer) {
								event = jse.libs.theme.events.ADD_CUSTOMIZER_WISHLIST();
							}
							url = options.wishlistUrl;
							break;
						case 'cart':
							if (customizer) {
								event = jse.libs.theme.events.ADD_CUSTOMIZER_CART();
								url = options.addCartCustomizerUrl;
							} else {
								url = options.addCartUrl;
							}
							break;
						case 'price_offer':
							$form.attr('action', options.priceOfferUrl).attr('method', options.priceOfferMethod);
							$form.off('submit');
							$form.submit();

							return;
						default:
							setTimeout(function () {
								$window.trigger(jse.libs.theme.events.STICKYBOX_CONTENT_CHANGE());
							}, 250);
							break;
					}

					if (event) {
						var deferred = $.Deferred();
						deferred.done(function (customizerRandom) {
							formdata[customizerRandom] = 0;
							_addToSomewhere(formdata, $form, url, $self);
						}).fail(function () {
							_addButtonState($self, '-fail');
						});
						$body.trigger(event, [{ 'deferred': deferred, 'dataset': formdata }]);
					} else if (url) {
						_addToSomewhere(formdata, $form, url, $self);
					}
				} else {
					_removeAddToCartThrobbler($form);
				}

				if (formdata.target === 'check') {
					previousModifiers = formdata.modifiers;
				}
			}).fail(function () {
				_removeAddToCartThrobbler($form);
				_addButtonState($self, '-fail');
			});
		}
	};

	/**
  * Handler for the change property option
  * */
	var _changeProductOptions = function _changeProductOptions(e) {
		var option = e.currentTarget;
		var optionValue = $(option).data('value');
		var optionContainer = $(option).parents('.modifier-group');

		$(optionContainer).find('li.active').removeClass('active');
		$(optionContainer).find('.modifier-item.active-modifier').removeClass('active-modifier');
		$(optionContainer).find('input.hidden-input').val(optionValue);
		$(optionContainer).find('input.hidden-input').trigger('blur', []);

		$(option).parents('li').addClass('active');
		$(option).addClass('active-modifier');
	};

	var _selectSelectedModifierInfo = function _selectSelectedModifierInfo(e) {
		var option = e.currentTarget;
		var price = $(option).attr('data-price');
		var label = $(option).attr('data-label');
		$(option).parents('.modifier-group').find('.selected-value-price').removeClass('temporary-value').attr('data-default-price', price);
		$(option).parents('.modifier-group').find('.selected-value').attr('data-default-value', label);
	};

	var _setSelectedModifierInfo = function _setSelectedModifierInfo(e) {
		var option = e.currentTarget;
		if (!$(option).parent().hasClass('active') && !$(option).is('select') && !$(option).hasClass('active-modifier')) {
			var price = $(option).attr('data-price');
			var label = $(option).attr('data-label');
			$(option).parents('.modifier-group').find('.selected-value-price').addClass('temporary-value').text(price);
			$(option).parents('.modifier-group').find('.selected-value').text(label);
		}
	};

	var _resetSelectedModifierInfo = function _resetSelectedModifierInfo(e) {
		var option = $(this);
		if (!$(option).parent().hasClass('active') && !$(option).is('select') && !$(option).hasClass('active-modifier')) {
			var priceHolder = $(option).parents('.modifier-group').find('.selected-value-price');
			var labelHolder = $(option).parents('.modifier-group').find('.selected-value');
			$(priceHolder).removeClass('temporary-value').text($(priceHolder).attr('data-default-price'));
			$(labelHolder).text($(labelHolder).attr('data-default-value'));
		}
	};

	/**
  * Keyup handler for quantity input field
  *
  * @param e
  * @private
  */
	var _keyupHandler = function _keyupHandler(e) {
		clearTimeout(timeout);

		timeout = setTimeout(function () {
			_submitHandler.call(this, e);
		}.bind(this), 300);
	};

	/**
  * Event handler for the add to cart button, that shows or hides the throbber.
  *
  * @param e
  */
	var _addToCartThrobberHandler = function _addToCartThrobberHandler(e) {
		var $btn = $this.find("[name=btn-add-to-cart]");
		var $btnFake = $this.find(".btn-add-to-cart-fake");
		var formReady = true;

		$(".properties-selection-form select").each(function () {
			var val = $(this).val();
			if (!val || val < 1) {
				formReady = false;
			}
		});

		if (formReady) {
			$btn.hide();
			$btnFake.show().prop("disabled", true).prepend('<span class="throbbler"></span>');
		}
	};

	/**
  * Cart dropdown oben event handler for the body.
  */
	var _cartDropdownOpenHandler = function _cartDropdownOpenHandler(e) {
		var $btn = $this.find("[name=btn-add-to-cart]");
		var $btnFake = $this.find(".btn-add-to-cart-fake");
		var fakeOrigLabel = $btnFake.html();
		var productCount = $(".cart-products-count").html();

		var textPhrases = JSON.parse($('#product-details-text-phrases').html());
		console.log(textPhrases['productsInCartSuffix']);

		$btnFake.html("<i class=\"fa fa-check\"></i> " + parseInt(productCount) + textPhrases['productsInCartSuffix']).prop("disabled", true).addClass("btn-buy-complete");

		setTimeout(function () {
			$btnFake.html(fakeOrigLabel).removeClass("btn-buy-complete").hide().prop("disabled", false);
			$(".throbbler", $btn).remove();
			$btn.show();
		}, 5000);
	};

	// ########## INITIALIZATION ##########

	/**
  * Init function of the widget
  * @constructor
  */
	module.init = function (done) {

		var $forms = $this.find('form');

		if (options.page === 'product-info') {
			$forms.on('submit', function (e) {
				_addToCartThrobberHandler(e);
			});
			$("body").on('CART_DROPDOWN_OPEN', _cartDropdownOpenHandler);
			$(document).on("mfpAfterClose", function (e) {
				_removeAddToCartThrobbler($forms);
			});
		}

		$forms.on('submit', { 'target': 'cart' }, _submitHandler).on('click', options.wishlistButtons, { 'target': 'wishlist' }, _submitHandler).on('click', options.priceOfferButtons, { 'target': 'price_offer' }, _submitHandler).on('change', options.attributes, { 'target': 'check' }, _submitHandler).on('mouseover', options.attributes, _setSelectedModifierInfo).on('mouseout', options.attributes, _resetSelectedModifierInfo).on('blur', options.productOptionField, { 'target': 'check' }, _submitHandler).on('click', options.productOptions, { 'target': 'check' }, function (e) {
			_selectSelectedModifierInfo(e);
			_changeProductOptions(e);
		}).on('mouseover', options.productOptions, _setSelectedModifierInfo).on('mouseout', options.productOptions, _resetSelectedModifierInfo).on('blur', options.quantity, { 'target': 'check' }, function (e) {
			_submitHandler(e);
		}).on('keyup', options.quantity, { 'target': 'check' }, _keyupHandler);

		// Fallback if the backend renders incorrect data
		// on initial page call
		$forms.not('.no-status-check').not('.product-info').each(function () {
			_submitHandler.call($(this));
		});
		done();
	};

	// Return data to widget engine
	return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
